package gov.va.vamf.scheduling.direct.eligibilitychecker;

import gov.va.vamf.scheduling.direct.domain.CustomRequestSetting;
import gov.va.vamf.scheduling.direct.domain.SchedulingDay;
import gov.va.vamf.scheduling.direct.domain.SchedulingDays;
import gov.va.vamf.scheduling.facility.timezone.FacilityTimezone;
import gov.va.vamf.scheduling.facility.timezone.FacilityTimezoneService;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import static gov.va.vamf.scheduling.direct.domain.CoreSettings.DEFAULT_CURRENT_TZ;
import static gov.va.vamf.scheduling.direct.domain.CoreSettings.DEFAULT_TIMEZONE;
import static gov.va.vamf.scheduling.direct.domain.CoreSettings.DEFAULT_TZ_OFFSET;
import static java.lang.Math.abs;

@Component
public class PatientClinicalServicesEligibilityChecker {

    @Resource
    FacilityTimezoneService facilityTimezoneService;

    private enum DayOfTheWeek {
        SUNDAY,
        MONDAY,
        TUESDAY,
        WEDNESDAY,
        THURSDAY,
        FRIDAY,
        SATURDAY;

        /**
         * Returns a DayOfTheWeek based on the numeric day of the week (where Sunday = 1, Monday = 2, etc.)
         * @param numericDayOfTheWeek
         * @return a DayOfTheWeek based on the numeric day of the week
         */
        public static DayOfTheWeek getDayOfTheWeek(final int numericDayOfTheWeek) {
            return DayOfTheWeek.values()[numericDayOfTheWeek - 1];
        }

        /**
         * Returns a numeric day of the week (where Sunday = 1, Monday = 2, etc.) for this day
         * @return a numeric day of the week for this day
         */
        public int getNumericDayOfTheWeek() {
            return this.ordinal() + 1;
        }
    }

    private SchedulingDay getSchedulingToday(SchedulingDays days, String timezone) {
        if (days == null) {
            return null;
        }

        final Calendar now = Calendar.getInstance(TimeZone.getTimeZone(timezone));
        final int today = now.get(Calendar.DAY_OF_WEEK);
        final DayOfTheWeek todayDayOfTheWeek = DayOfTheWeek.getDayOfTheWeek(today);

        // Ideally this would be:
        //    int index = today - 1;
        //    SchedulingDay day = days.get(index)
        //
        // But because there is Scheduling Days has no assurances of order of the array this loop is required
        for (final SchedulingDay day : days) {
            final String dayOfTheWeek = day.getDay().toString();
            if (DayOfTheWeek.valueOf(dayOfTheWeek) == todayDayOfTheWeek) {
                return day;
            }
        }

        return null;
    }

    private String defaultTimeZone(FacilityTimezone facilityTimezone) {
        if (facilityTimezone == null) {
            return DEFAULT_TIMEZONE;
        }
        final String timezone = facilityTimezone.getTimezone();
        return timezone != null ? timezone : DEFAULT_TIMEZONE;
    }

    private String defaultCurrentTimeZone(FacilityTimezone facilityTimezone) {
        if (facilityTimezone == null) {
            return DEFAULT_CURRENT_TZ;
        }
        final String timezone = facilityTimezone.getCurrentTZ();
        return timezone != null ? timezone : DEFAULT_CURRENT_TZ;
    }


    private String getTimeZoneOffset(FacilityTimezone timezone) {
        if (timezone == null || timezone.getTimezone() == null) {
            return DEFAULT_TZ_OFFSET;
        }

        final TimeZone tz = TimeZone.getTimeZone(timezone.getTimezone());
        final int milliseconds = tz.getOffset(new Date().getTime());
        final int seconds = milliseconds / 1000;
        final int min = seconds / 60;
        final int hours = abs(min / 60);
        final boolean isNegative = milliseconds < 0;

        if (isNegative) {
            return String.format("-%02d:00", hours);
        }
        return String.format("+%02d:00", hours);
    }

    public ExpressCareTimes getExpressCareTimes(String siteCode, CustomRequestSetting customRequestSetting) {
        if (customRequestSetting == null) {
            return null;
        }
        final FacilityTimezone facilityTimezone = facilityTimezoneService.fetchFacilityTimezoneById(siteCode);
        final String timeZone = this.defaultTimeZone(facilityTimezone);
        final String currentTimezone = this.defaultCurrentTimeZone(facilityTimezone);
        final String offsetUtc = this.getTimeZoneOffset(facilityTimezone);
        final SchedulingDays days = customRequestSetting.getSchedulingDays();
        final SchedulingDay day = this.getSchedulingToday(days, timeZone);

        if (day != null && day.getCanSchedule()) {
            ExpressCareTimes expressCareTimes = new ExpressCareTimes();

            expressCareTimes.setStart(day.getStartTime());
            expressCareTimes.setEnd(day.getEndTime());
            expressCareTimes.setTimezone(currentTimezone);
            expressCareTimes.setOffsetUtc(offsetUtc);

            return expressCareTimes;
        }
        return null;
    }

    /**
     * Determine if express care booking is supported at this moment for the given site and express care configuration.

     * @param siteCode
     * @param customRequestSetting
     * @return
     */
    public ExpressCareTimeValidityAndDetail isCustomRequestSupportedNow(String siteCode, CustomRequestSetting customRequestSetting) {

        FacilityTimezone facilityTimezone = facilityTimezoneService.fetchFacilityTimezoneById(siteCode);

        String tz = (facilityTimezone != null && facilityTimezone.getTimezone() != null) ?
                facilityTimezone.getTimezone() : DEFAULT_TIMEZONE;

        String facilityCurrentTz = (facilityTimezone != null && facilityTimezone.getCurrentTZ() != null) ?
                facilityTimezone.getCurrentTZ() : DEFAULT_CURRENT_TZ;

        if(customRequestSetting != null && customRequestSetting.getSupported()) {
            SchedulingDays schedulingDays = customRequestSetting.getSchedulingDays();
            if(schedulingDays != null) {
                for(SchedulingDay day : schedulingDays) {
                    if(day.getCanSchedule()) {
                        String dayOfWeek = day.getDay().toString();

                        Calendar startCalendar = CalendarHelper.createCalendar(tz, dayOfWeek, day.getStartTime());
                        Calendar endCalendar   = CalendarHelper.createCalendar(tz, dayOfWeek, day.getEndTime());
                        Calendar now = Calendar.getInstance(TimeZone.getTimeZone(tz));

                        if(now.after(startCalendar) && now.before(endCalendar)) {
                            return new ExpressCareTimeValidityAndDetail(ExpressCareTimeValidity.VALID);
                        } else if(CalendarHelper.dayOfWeekMap.get(dayOfWeek).intValue() == now.get(Calendar.DAY_OF_WEEK)) {
                            final String expressCareEndTime = day.getEndTime() + " " + facilityCurrentTz;
                            return new ExpressCareTimeValidityAndDetail(ExpressCareTimeValidity.INVALID_TIME, expressCareEndTime);
                        }
                    }
                }
            }
        }

        return new ExpressCareTimeValidityAndDetail(ExpressCareTimeValidity.INVALID_DAY);
    }

    public enum ExpressCareTimeValidity {
        VALID,
        // TODO: the middle tier should really just be providing an error code, and the front end providing the text
        INVALID_DAY("Express Care requests cannot be submitted today. If you still wish to submit a request for a " +
                "visit, please change your Type of Care selection. For urgent matters please contact your facility."),
        INVALID_TIME("Express Care requests cannot be submitted after %s today. If you still wish to submit a " +
                "request for a visit, please change your Type of Care selection. For urgent matters please contact " +
                "your facility.");

        private String message;

        ExpressCareTimeValidity() {
        }

        ExpressCareTimeValidity(final String message) {
            this.message = message;
        }

        public String getMessage() {
            return this.message;
        }
    }

    public static class ExpressCareTimeValidityAndDetail {
        private final ExpressCareTimeValidity validity;
        private final String detail;

        public ExpressCareTimeValidityAndDetail(final ExpressCareTimeValidity expressCareTimeValidity) {
            this(expressCareTimeValidity, null);
        }

        public ExpressCareTimeValidityAndDetail(final ExpressCareTimeValidity validity, final String detail) {
            this.validity = validity;
            this.detail = detail;
        }

        public ExpressCareTimeValidity getValidity() {
            return validity;
        }

        public String getDetail() {
            return detail;
        }

        public String formatMessage() {
            final String message = this.validity.getMessage();
            if (message == null) {
                return null;
            }

            return String.format(message, this.detail);
        }

    }
}
